#include <sys/stat.h>
#include <cerrno>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <vector>
using namespace std;
#include <fcntl.h>
#include <unistd.h>
#include <sys/soundcard.h>
#include "dsf.h"
#include "dsf_load.h"
unsigned char
bit_swap(unsigned char c)
{
	char	r;
	r = (c << 4) | (c >> 4);
	r = ((r & 0x33) << 2) | ((r & 0xcc) >> 2);
	r = ((r & 0x55) << 1) | ((r & 0xaa) >> 1);
	return	r;
}
int
skip_dsd( istream &is,
		int channels, int blocksz, u_int64_t samples, bool msbfirst)
{
	for (long long blkno = 0; blkno * channels * blocksz < samples; blkno++) {
		is.ignore(channels*blocksz);
		if (is.gcount() != channels*blocksz) {
			cerr << __func__
				 << ": block(" << blkno << ") can't read block, read byte="
				 << is.gcount() << endl;
			return	1;
		}
	}
	return	0;
}
int
skip_dsf_stream( const char *filename, istream &is)
{
	bool	r;
	dsf_t	dsf;
	r	= dsf_load(dsf, is);
	if (!r) {
		cerr << filename << ": dsf load error." << endl;
		return	1;
	}
	if (skip_dsd(
				is,
				dsf.channelnum,
				dsf.blocksizeperchannel,
				dsf.datachunksize-12,
				dsf.bitspersample==8)) {
		cerr << filename << ": play dsd failed." << endl;
		return	1;
	}
	streamsize	skipsize = 
			dsf.filesize
			- dsf.dsfchunksize - dsf.fmtchunksize - dsf.datachunksize;
	is.ignore( skipsize);
	if (is.gcount() != skipsize)
		return	1;
	return	0;
}
bool
play_dsd(
		int fd, istream &is,
		int channels, int blocksz, u_int64_t samples, bool msbfirst, int sz)
{
	vector<char>	rdbuf(channels*blocksz);
	vector<char>	plbuf(channels*blocksz);
	for (long long blkno = 0; blkno * channels * blocksz < samples; blkno++) {
		is.read(rdbuf.data(), rdbuf.size());
		if (is.gcount() != rdbuf.size()) {
			cerr << __func__
				 << ": block(" << blkno << ") can't read block, read byte="
				 << is.gcount() << endl;
			return	false;
		}
		for (int i=0; i< blocksz/4; i++) {
			int	rdofs	= i*4;
			for (int c=0; c< channels; c++) {
				int	blockofs = rdofs + c*blocksz;
				int	plofs	= (i*channels+c)*4;
				if (msbfirst) {
					*(u_int32_t*)&plbuf[plofs]= *(u_int32_t*)&rdbuf[blockofs];
				}
				else {
					for (int b=0; b< sizeof(u_int32_t); b++) {
						plbuf[plofs+b]	= bit_swap(rdbuf[blockofs+b]);
					}
				}
			}
		}
		if (sz == plbuf.size()) {
			if (write(fd, plbuf.data(), plbuf.size()) != plbuf.size()) {
				cerr << __func__
					 << ": write failed, " << strerror(errno)
					 << endl;
				return	false;
			}
		}
		else {
			int i;
			for (i=0; (i + sz) < plbuf.size(); i+=sz) {
				if (write(fd, plbuf.data()+i, sz) != sz) {
					cerr << __func__
						 << ": write failed, " << strerror(errno)
						 << endl;
					return	false;
				}
			}
			int count = plbuf.size() % sz;
			if (write(fd, plbuf.data()+i, count) != count) {
				cerr << __func__
					 << ": write failed, " << strerror(errno)
					 << endl;
				return	false;
			}
		}
	}
	return	true;
}
int
play_dsf_stream(
	const char *devname, const char *filename, istream &is,
	unsigned long uwaittime, bool quit)
{
	bool	r;
	dsf_t	dsf;
	r	= dsf_load(dsf, is);
	if (!r) {
		cerr << filename << ": dsf load error." << endl;
		return	1;
	}
	int		freq	= dsf.frequency / 32;
	if (!devname)
		devname= getenv("DSP");
	if (!devname)
		devname	= "/dev/dsp";
	int	fd, sz;
		sz	= dsf.blocksizeperchannel * dsf.channelnum;
	if (strcmp(devname,"-")==0) {
		fd	= 1;
	}
	else {
		fd	= open(devname, O_WRONLY);
		if (fd < 0) {
			cerr << devname << ": can't open device." << endl;
			return	1;
		}
		int	c;
		c	= AFMT_DSD;
		if (ioctl(fd, SNDCTL_DSP_SETFMT, &c)) {
			cerr << devname << ": not support dsd format in device." << endl;
			return	1;
		}
		c	= freq;
		if (ioctl(fd, SNDCTL_DSP_SPEED, &c)) {
			cerr << devname << ": not support frequency(" << c << ")." << endl;
			return	1;
		}
		c	= dsf.channelnum;
		if (ioctl(fd, SNDCTL_DSP_CHANNELS, &c)) {
			cerr << devname << ": not support channels(" << c << ")." << endl;
			return	1;
		}
		// if (ioctl(fd, SNDCTL_DSP_SETBLKSIZE, &sz)) {
		// 	cerr << devname << ": setblksize failed." << endl;
		// 	return	1;
		// }
		if (ioctl(fd, SNDCTL_DSP_GETBLKSIZE, &sz)) {
			cerr << devname << ": getblksize failed." << endl;
			return	1;
		}
	}
	if (quit)
		return	1;
	if (uwaittime)
		usleep(uwaittime);
	u_int64_t	samples	= dsf.datachunksize-12;
	bool	msbfirst	= dsf.bitspersample!=8;
	// cerr << "msbfirst: " << msbfirst << endl;
	r	= play_dsd(
				fd, is,
				dsf.channelnum,
				dsf.blocksizeperchannel,
				samples,
				msbfirst, sz);
	if (!r) {
		cerr << filename << ": play dsd failed." << endl;
		close(fd);
		return	1;
	}
	// u_int64_t	usleeptime = samples / dsf.channelnum / 4 * 1000000 / freq;
	// cerr << "usleeptime: " << usleeptime << endl;
	// usleep(usleeptime);
	close(fd);
	vector<char>	skipbuf(
			dsf.filesize
			- dsf.dsfchunksize - dsf.fmtchunksize - dsf.datachunksize);
	is.read( skipbuf.data(), skipbuf.size());
	if (is.gcount() != skipbuf.size())
		return	1;
	return	0;
}
int main( int argc, char **argv)
{
	if (argc <= 1) {
		cout << "usage: " << argv[0] << " [-d] [-q] [-a <devname>] [-w waittime] <dsf file...>"
			 << endl;
		return	0;
	}
	bool	debug=false;
	bool	quit=false;
	const char *devname=NULL;
	unsigned long uwaittime=0;
	for (int i=1; i< argc; i++) {
		if (strcmp(argv[i], "-d")==0) {
			debug=true;
		}
		else if (strcmp(argv[i], "-a")==0) {
			if (++i<argc)
				devname = argv[i];
			else
				break;
		}
		else if (strcmp(argv[i], "-w")==0) {
			if (++i<argc)
				uwaittime = strtoul(argv[i], NULL, 0)*1000000;
			else
				break;
		}
		else if (strcmp(argv[i], "-q")==0) {
			quit= true;
		}
		else if (strcmp(argv[i], "-")==0) {
			const char *filename = "(stdin)";
			int	r;
			if (debug)
				r	= skip_dsf_stream( filename, cin);
			else
				r	= play_dsf_stream( devname, filename, cin, uwaittime, quit);
			if (r) {
				cerr << filename << ": " << i << ": play dsf failed." << endl;
				return	1;
			}
		}
		else {
			const char *filename = argv[i];
			ifstream	ifs;
			ifs.open(filename);
			if (!ifs) {
				cerr << filename << ": " << strerror(errno) << endl;
				return	1;
			}
			int	r;
			if (debug)
				r	= skip_dsf_stream( filename, ifs);
			else
				r	= play_dsf_stream( devname, filename, ifs, uwaittime, quit);
			if (r) {
				cerr << filename << ": play dsf failed." << endl;
				return	1;
			}
			// ifs.close();
			// if (debug) {
			// 	struct stat sb;
			// 	if (stat(filename, &sb)) {
			// 		cerr << filename << ": stat failed, "
			// 			 << strerror(errno) << endl;
			// 	}
			// 	cout << "sb.st_size: " << sb.st_size << endl;
			// 	cout << "tellg: " << ifs.tellg() << endl;
			// }
		}
	}
	return 0;
}
